unit ProcessCheckerSettings;
{*******************************************************************************
  ProcessChecker Demo
  Written by David Clegg, davidclegg@optusnet.com.au.

  This unit contains the ProcessCheckerSettings class, which is used to store
  the settings for the application.
*******************************************************************************}

interface

uses
  ProcessClasses, System.Runtime.Serialization;

type
	/// <summary>
	/// Class to persist the settings for ProcessChecker. It is marked with the
	/// [Serializable] attribute so we can serialize it to disk. All other
	/// classes exposed as properties of this class must also be marked as
	/// [Serializable], or have their field declarations marked as
	/// [NonSerialized].
	/// </summary>
  [Serializable]
  TProcessCheckerSettings = class
  strict private
    FWatchedProcesses: TProcesses;
    FCheckFrequency: integer;
    FRestartOneProcess: boolean;
    FEnabled: boolean;
		[NonSerialized]
    FSettingsLoaded: EventHandler;
		[NonSerialized]
    FSettingsSaved: EventHandler;
    procedure InitDefaults;
  public
    property WatchedProcesses: TProcesses read FWatchedProcesses;
    property CheckFrequency: integer read FCheckFrequency write FCheckFrequency;
    property RestartOneProcess: boolean read FRestartOneProcess write FRestartOneProcess;
    property SettingsLoaded: EventHandler add FSettingsLoaded remove FSettingsLoaded;
    property SettingsSaved: EventHandler add FSettingsSaved remove FSettingsSaved;
    property Enabled: boolean read FEnabled write FEnabled;
    class function Load: TProcessCheckerSettings;
    procedure Save;
    constructor Create;
  end;

implementation

uses
  System.Runtime.Serialization.Formatters.Binary, System.Windows.Forms,
  System.IO;

const
  CHECK_FREQUENCY = 5;
  RESTART_ONE_PROCESS = True;
  SETTINGS_EXTENSION = 'dat';

constructor TProcessCheckerSettings.Create;
begin
  inherited Create;
  FWatchedProcesses := TProcesses.Create;
  InitDefaults;
end;

/// <summary>
/// Serialize a ProcessCheckerSettings instance from disk.
/// </summary>
class function TProcessCheckerSettings.Load: TProcessCheckerSettings;
var
  lSettings: TProcessCheckerSettings;
  lFileName: string;
  lFileStream: FileStream;
  lFormatter: BinaryFormatter;
begin
  //The ProcessCheckerSettings will be deserialized from .\ProcessChecker.dat
  lFileName := Path.ChangeExtension(Application.ExecutablePath, SETTINGS_EXTENSION);
  if (System.IO.File.Exists(lFileName)) then
  begin
    lFileStream := FileStream.Create(lFileName, FileMode.Open);
    try
      lFormatter := BinaryFormatter.Create;
      lSettings := TProcessCheckerSettings(lFormatter.Deserialize(lFileStream));
    finally
      lFileStream.Close;
    end;
  end
  else
    lSettings := TProcessCheckerSettings.Create;


  //Fire the event handler(s) if any delegates have been assigned.
  if Assigned(lSettings.FSettingsLoaded) then
    lSettings.FSettingsLoaded(lSettings, nil);

  Result := lSettings;
end;

/// <summary>
/// Serialize the ProcessCheckerSettings instance to disk.
/// </summary>
procedure TProcessCheckerSettings.Save;
var
  lFileName: string;
  lFileStream: FileStream;
  lFormatter: BinaryFormatter;
begin
  //The instance will be serialized to .\ProcessChecker.dat
  lFileName := Path.ChangeExtension(Application.ExecutablePath, SETTINGS_EXTENSION);
  lFileStream := FileStream.Create(lFileName, FileMode.Create);
  try
    //Create the FileStream to stream out the .dat file, and
    //serialize using the BinaryFormatter class.
    lFormatter := BinaryFormatter.Create;
    lFormatter.Serialize(lFileStream, Self);
  finally
    lFileStream.Close();
  end;
  //Fire the event handler(s) if any delegates have been assigned.
  if Assigned(FSettingsSaved) then
    FSettingsSaved(Self, nil);
end;

procedure TProcessCheckerSettings.InitDefaults;
begin
  FCheckFrequency := CHECK_FREQUENCY;
  FRestartOneProcess := RESTART_ONE_PROCESS;
end;

end.


